/*jshint eqeqeq:false, strict:false*/
/*global box:false*/
box.store('ysl').addModule('parallax-easing', function() {
    return {
        /*!
         * Easing Equations v2.0
         * (c) 2003 Robert Penner, all rights reserved.
         * This work is subject to the terms in http://www.robertpenner.com/easing_terms_of_use.html
         */
        linear: function(nFrom, nTo, nScroll) {
            return (nTo - nFrom) * nScroll + nFrom;
        },
        quadIn: function(nFrom, nTo, nScroll) {
            return (nTo - nFrom) * nScroll * nScroll + nFrom;
        },
        quadOut: function(nFrom, nTo, nScroll) {
            return - (nTo - nFrom) * nScroll * (nScroll - 2) + nFrom;
        },
        quadInOut: function(nFrom, nTo, nScroll) {
            nScroll *= 2;
            return nScroll < 1 ?
                (nTo - nFrom) / 2 * nScroll * nScroll + nFrom :
                - (nTo - nFrom) / 2 * (--nScroll * (nScroll - 2) - 1) + nFrom;
        },
        cubicIn: function(nFrom, nTo, nScroll) {
            return (nTo - nFrom) * Math.pow(nScroll, 3) + nFrom;
        },
        cubicOut: function(nFrom, nTo, nScroll) {
            return (nTo - nFrom) * (Math.pow(nScroll - 1, 3) + 1) + nFrom;
        },
        cubicInOut: function(nFrom, nTo, nScroll) {
            nScroll *= 2;
            return nScroll < 1 ?
                (nTo - nFrom) / 2 * Math.pow(nScroll, 3) + nFrom :
                (nTo - nFrom) / 2 * (Math.pow(nScroll - 2, 3) + 2) + nFrom;
        },
        sineIn: function(nFrom, nTo, nScroll) {
            return (nTo - nFrom) * (1 - Math.cos(nScroll * Math.PI / 2)) + nFrom;
        },
        sineOut: function(nFrom, nTo, nScroll) {
            return (nTo - nFrom) * Math.sin(nScroll * Math.PI / 2) + nFrom;
        },
        sineInOut: function(nFrom, nTo, nScroll) {
            return (nTo - nFrom) / 2 * (1 - Math.cos(nScroll * Math.PI)) + nFrom;
        },
        circOut: function(nFrom, nTo, nScroll) {
            nScroll--;
            return (nTo - nFrom) * Math.sqrt(1 - nScroll * nScroll) + nFrom;
        }
    };
});

box.store('ysl').addConfig('parallax-item', {
    speedToPx: 80
}).addConstructor('parallax-item', function($, box) {
    var reCheckSpeed = /^speed\((-?\d+)\)$/,
        reCheckScrollingInView = /^inview\(((?:1|0|(?:0\.\d+))) ([1-9]+)\)$/,
        reCheckScrollingFrom = /^from\(/,
        reCheckScrollingTo = /to\([^\)]+\)$/,
        reParseScrolling = /(?:from|to|step)\([^\)]+\)/g,
        reParseScrollingStep = /^[a-z]+\(([a-z0-9+-]+)\s+((?:1|0|(?:0\.\d+)))(?:\s+([a-zA-Z]+))?\)$/,
        reParseViewportFn = /^viewport(before|start|middle|end|after).*$/,
        reGetViewportOffset = /^[a-z-]+((?:\+|-)\d+)$/,
        
        oEasing = box.get('ysl:parallax-easing'),
        
        parseSpeed, parseInView, parseFromTo, parseFromStepToValues, getTopFromView,
        
        oProto;
    
    parseSpeed = function(oCom, sValue) {
        // parse a string like "speed(5)"
        var nSpeed = parseInt(sValue.match(reCheckSpeed)[1], 10),
            nTop = oCom.getTop(),
            nHeight = oCom.getHeight(),
            nPx = oCom.cfg.speedToPx * nSpeed,
            nScrollStart = Math.max(0, (nTop - oCom.viewHeight) / (oCom.pageHeight - oCom.viewHeight)),
            nScrollEnd = Math.min(1, Math.abs(nTop - nPx + nHeight) / (oCom.pageHeight - oCom.viewHeight));
        oCom.data = [
            {
                easing: oEasing.linear,
                original: sValue,
                scroll: nScrollStart,
                speed: nSpeed,
                top: nTop
            }, {
                easing: oEasing.linear,
                original: sValue,
                scroll: nScrollEnd,
                speed: nSpeed,
                top: nTop - nPx
            }
        ];
    };
    
    parseInView = function(oCom, sValue) {
        // parse a string like "inview(0.04 1)"
        var aMatch = sValue.match(reCheckScrollingInView),
            nSpeed = parseInt(aMatch[2], 10) + 1,
            nPx = oCom.cfg.speedToPx * nSpeed,
            nTop = oCom.getTop(),
            nHeight = oCom.getHeight(),
            nScrollFactor = parseFloat(aMatch[1]),
            nOffset = (oCom.viewHeight - nHeight) / 2,
            nInView = nScrollFactor * (oCom.pageHeight - oCom.viewHeight) + nOffset;
        oCom.data = [
            {
                easing: oEasing.linear,
                original: sValue,
                scroll: 0,
                top: nInView + nScrollFactor * nPx
            }, {
                easing: oEasing.linear,
                original: sValue,
                scroll: 1,
                top: nInView - (1 - nScrollFactor) * nPx
            }
        ];
    };
    
    parseFromTo = function(oCom, sValue) {
        // parse a string like "from(520 0), step(250 0.5), to(0 1)"
        var aMatch = sValue.match(reParseScrolling),
            i, l;
        if(aMatch) {
            oCom.data = [];
            i = -1;
            l = aMatch.length;
            while(++i < l) {
                this.data[i] = parseFromStepToValues(aMatch[i]);
                if(oCom.invalid === true) {
                    return;
                }
            }
        }
    };
    
    parseFromStepToValues = function(oCom, sValue) {
        // parse a string like "from(520 0)" or "step(250 0.5)" or "to(0 1)"
        var oData = { original: sValue },
            aTmp = sValue.match(reParseScrollingStep),
            nTop;
        if(aTmp) {
            oData.scroll = parseFloat(aTmp[2]);
            nTop = parseInt(aTmp[1], 10);
            if(isNaN(nTop)) {
                nTop = getTopFromView(oCom, aTmp[1], oData.scroll);
                if(isNaN(nTop)) {
                    oCom.boxPublish('error', { message: 'bad position in "' + sValue + '"' });
                    oCom.invalid = true;
                    return;
                }
            }
            oData.top = nTop;
            oData.easing = oEasing[aTmp[3]] || oEasing.linear;
        } else {
            oCom.boxPublish('error', { message: 'bad step declaration with "' + sValue + '"' });
            oCom.invalid = true;
        }
        return oData;
    };
    
    getTopFromView = function(oCom, sValue, nScroll) {
        var nOffset = parseInt(sValue.replace(reGetViewportOffset, '$1'), 10) || 0,
            sKey = sValue.replace(reParseViewportFn, '$1');
        if(sKey === 'before') {
            return nScroll * oCom.maxScroll - oCom.getHeight() + nOffset;
        } else if(sKey === 'start') {
            return nScroll * oCom.maxScroll + nOffset;
        } else if(sKey === 'middle') {
            return nScroll * oCom.maxScroll + (oCom.viewHeight - oCom.getHeight()) / 2 + nOffset;
        } else if(sKey === 'end') {
            return nScroll * oCom.maxScroll + oCom.viewHeight - oCom.getHeight() + nOffset;
        } else if(sKey === 'after') {
            return nScroll * oCom.maxScroll + oCom.viewHeight + nOffset;
        }
    };
    
    oProto = {
        boxCreate: function(oData) {
            this.rootElm = oData.rootElm;
            this.originalTop = this.rootElm.style.top;
            this.index = oData.index;
            this.pageHeight = oData.pageHeight;
            this.viewHeight = oData.viewHeight;
            this.maxScroll = oData.maxScroll;
            this.invalid = false;
            this.parseScrolling();
        },

        boxDestroy: function() {
            this.rootElm.style.top = this.originalTop;
        },
        
        getTop: function() {
            return parseInt(this.rootElm.style.top, 10);
        },
        
        getHeight: function() {
            return this.rootElm.offsetHeight;
        },
        
        parseScrolling: function() {
            var sValue = this.rootElm.getAttribute('data-scrolling');
            if(sValue == this.scrolling) {
                return;
            }
            this.scrolling = sValue;
            if(typeof sValue === 'string' && sValue) {
                if(reCheckSpeed.test(sValue)) {
                    parseSpeed(this, sValue);
                } else if(reCheckScrollingInView.test(sValue)) {
                    parseInView(this, sValue);
                } else if(reCheckScrollingFrom.test(sValue) && reCheckScrollingTo.test(sValue)) {
                    parseFromTo(this, sValue);
                } else {
                    this.invalid = true;
                    this.boxPublish('error', { message: 'data-scrolling attribute is not readable ("' + sValue + '")' });
                }
            } else {
                this.invalid = true;
                this.boxPublish('warning', { message: 'data-scrolling attribute is absent' });
            }
        },
        
        getDataIndex: function(nScrollFactor) {
            if(this.invalid === false) {
                var aData = this.data,
                    i = -1,
                    l = aData.length - 1;
                while(++i < l) {
                    if(nScrollFactor > aData[i].scroll && nScrollFactor <= aData[i + 1].scroll) {
                        return i;
                    }
                }
                return nScrollFactor <= aData[0].scroll ? 'first' : 'last';
            }
        },
        
        move: function(nScrollFactor) {
            if(this.invalid === false) {
                var uIndex = this.getDataIndex(nScrollFactor),
                    aData = this.data,
                    nTopNew;
                if(uIndex === 'first') {
                    nTopNew = aData[0].top;
                } else if(uIndex === 'last') {
                    nTopNew = aData[aData.length - 1].top;
                } else {
                    var oCurrent = aData[uIndex],
                        oNext = aData[uIndex + 1],
                        nScrollFrom = oCurrent.scroll,
                        nScrollTo = oNext.scroll,
                        nNewFactor = (nScrollFactor - nScrollFrom) / (nScrollTo - nScrollFrom),
                        nTopFrom = oCurrent.top,
                        nTopTo = oNext.top;
                    nTopNew = oNext.easing(nTopFrom, nTopTo, nNewFactor);
                }
                this.rootElm.style.top = nTopNew + 'px';
            }
        }
    };
    
    return box.get('util:component').create({
        extend: oProto
    });
});

box.store('ysl').addConstructor('parallax', function($, box) {
    var oProto = {
        boxCreate: function(oData) {
            this.rootElm = $(oData.rootElm);
            this.itemsSelector = oData.itemsElm;
        },

        addItem: function(oElm) {
            var oCom = this,
                i = oCom.items.length;
            oCom.items[i] = box.get('ysl').create('parallax-item.n' + i, {
                rootElm: oElm,
                index: i,
                pageHeight: oCom.pageHeight,
                viewHeight: oCom.viewHeight,
                maxScroll: oCom.maxScroll
            });
            return oCom.items[i];
        },

        removeItem: function(i) {
            return this.items.splice(i, 1);
        },
        
        activate: function(oData) {
            var oCom = this;
            oCom.pageHeight = oData.pageHeight;
            oCom.viewHeight = oData.viewHeight;
            oCom.maxScroll = oData.maxScroll;
            oCom.scrollTop = oData.scrollTop;
            oCom.items = [];
            oCom.rootElm.find(oCom.itemsSelector).each(function(i, oElm) {
                oCom.addItem(oElm);
            });
            this.rootElm.addClass('parallax');
            this.activated = true;
            this.move(oData.scrollFactor);
        },
        
        deactivate: function() {
            this.rootElm.removeClass('parallax');
            var i = this.items.length;
            while(i--) {
                box.get('ysl').destroy('parallax-item.' + this.items[i].id);
            }
            this.items = null;
            this.activated = false;
        },
        
        move: function(nScrollFactor) {
            if(this.activated === true) {
                var i = this.items.length;
                while(i--) {
                    this.items[i].move(nScrollFactor);
                }
            }
        }
    };
    
    return box.get('util:component').create({
        extend: oProto
    });
});

box.store('ysl').addConfig('page-scroll', {
    htmlScrollbar: '<div class="{$clsScrollbar}"><div class="{$clsGutter}"><span class="{$clsFace}"{$styleFace}></span></div></div>',
    clsScrollbar: 'page-scrollbar',
    clsGutter: 'page-scrollbar-gutter',
    clsFace: 'page-scrollbar-face',
    clsNoSupport: 'no-page-scroll',
    moveBy: 150,
    altMoveBy: 400,
    threshold: 100,
    frameRate: 40,
    maxAnimFrames: 50,
    minAnimFrames: 15
}).addConstructor('page-scroll', function($, box) {
    var bTouch = false,
        reFocusable = /^(?:A|BUTTON|INPUT)$/,
        oEasing = box.get('ysl:parallax-easing'),
        nTimerScroll, nTimerResize;

    function onDraggableMove(oEvt) {
        this.moveScrollbarTo(oEvt.data.top, false);
    }

    function addDraggable(oCom) {
        var sId = 'draggable.page-scroll-' + oCom.id;
        oCom.draggable = box.get('ui').create(sId, {
            rootElm: oCom.faceElm,
            limits: {
                minLeft: 0, maxLeft: 0,
                minTop: 0, maxTop: oCom.maxFaceOffset
            }
        });
        box.subscribe({
            name: 'move>ui:' + sId,
            context: oCom,
            handler: onDraggableMove
        });
    }

    function onGutterClick(oCom) {
        return function(oEvt) {
            if(oEvt.target === this) {
                oCom.moveScrollbarTo(oEvt.pageY - oCom.faceHeight / 2);
            }
        };
    }

    function addScrollbar(oCom) {
        var nTop = Math.round(oCom.getOffset() / oCom.maxContentOffset * oCom.maxFaceOffset),
            oCfg = oCom.cfg,
            sHtml = oCfg.htmlScrollbar
                .replace('{$clsScrollbar}', oCfg.clsScrollbar)
                .replace('{$clsGutter}', oCfg.clsGutter)
                .replace('{$clsFace}', oCfg.clsFace)
                .replace('{$styleFace}', ' style="top:' + nTop + 'px; height:' + (oCom.faceHeight - oCom.faceBorders) + 'px;"');
        oCom.scrollElm = $(sHtml).insertAfter(oCom.rootElm);
        oCom.gutterElm = oCom.scrollElm.find('.' + oCfg.clsGutter).click(onGutterClick(oCom));
        oCom.faceElm = oCom.scrollElm.find('.' + oCfg.clsFace);
        addDraggable(oCom);
    }

    function removeScrollbar(oCom) {
        var sId = 'draggable.page-scroll-' + oCom.id;
        box.unsubscribe('move>ui:' + sId);
        box.get('ui').destroy(sId);
        oCom.scrollElm.remove();
        oCom.scrollElm = oCom.gutterElm = oCom.faceElm = null;
    }

    function onDocScroll(oCom) {
        return function(oEvt) {
            oEvt.preventDefault();
            var nTime = oEvt.timeStamp,
                nTimeDiff = nTime - oCom.timeStamp,
                nMovePlus = nTimeDiff < oCom.cfg.threshold ? oCom.cfg.threshold - nTimeDiff : 0,
                n = oEvt.detail ? oEvt.detail / 3 : - oEvt.wheelDelta / 120,
                nTop = oCom.topContent + (n * (oCom.cfg.moveBy + nMovePlus));
            oCom.timeStamp = nTime;
            oCom.moveContentTo(Math.max(0, nTop));
        };
    }

    function disableOnFirstTouch(oCom) {
        $('body').bind('touchstart.moodboards', function() {
            bTouch = true;
            oCom.disable();
            $(box.getDoc().documentElement).addClass(oCom.cfg.clsNoSupport);
            $(this).unbind('touchstart.moodboards');
            oCom.boxPublish('nosupport');
        });
    }

    function setOffsetTop(oCom, nTop) {
        if(oCom.isMovingContent) {
            oCom.topContent = Math.round(nTop);
            oCom.topFace = Math.round(nTop / oCom.maxContentOffset * oCom.maxFaceOffset);
        } else {
            oCom.topFace = Math.round(nTop);
            oCom.topContent = Math.round(nTop / oCom.maxFaceOffset * oCom.maxContentOffset);
        }
        oCom.rootElm[0].style.top = '-' + oCom.topContent + 'px';
        oCom.faceElm[0].style.top = oCom.topFace + 'px';
        oCom.parallax.move(oCom.getScrollFactor());
    }

    function setOffsetTopInTimer(oCom, bContent) {
        return function() {
            var sFrom = bContent ? 'topContentFrom' : 'topFaceFrom',
                sTo = bContent ? 'topContentTo' : 'topFaceTo',
                nTop = oEasing.circOut(oCom[sFrom], oCom[sTo], oCom.currentFrame / oCom.totalFrame);
            setOffsetTop(oCom, nTop);
            if(++oCom.currentFrame > oCom.totalFrame) {
                box.getWin().clearInterval(nTimerScroll);
                nTimerScroll = null;
            }
        };
    }

    function getFrameNumber(oCom, bContent) {
        var oCfg = oCom.cfg,
            nMoveBy = oCfg.moveBy,
            sFrom = bContent ? 'topContentFrom' : 'topFaceFrom',
            sTo = bContent ? 'topContentTo' : 'topFaceTo',
            nSlope = (oCfg.maxAnimFrames - oCfg.minAnimFrames) / (oCom[bContent ? 'maxContentOffset' : 'maxFaceOffset'] - nMoveBy);
        return Math.round(nSlope * Math.abs(oCom[sTo] - oCom[sFrom]) + (oCfg.minAnimFrames - nSlope * nMoveBy));
    }

    function computeFrames(oCom) {
        oCom.totalFrame = getFrameNumber(oCom, oCom.isMovingContent);
        oCom.currentFrame = 1;
        if(!nTimerScroll) {
            nTimerScroll = box.getWin().setInterval(setOffsetTopInTimer(oCom, oCom.isMovingContent), oCom.cfg.frameRate);
        }
    }

    function onDocKeyDown(oCom) {
        return function(oEvt) {
            if(reFocusable.test(oEvt.target.tagName)) {
                return;
            }
            var nKey = oEvt.which;
            switch(nKey) {
                case 32:
                case 34:
                    oEvt.preventDefault();
                    oCom.moveContentTo(-oCom.getOffset() + oCom.cfg.altMoveBy);
                    break;
                case 33:
                    oEvt.preventDefault();
                    oCom.moveContentTo(-oCom.getOffset() - oCom.cfg.altMoveBy);
                    break;
                case 35:
                    oEvt.preventDefault();
                    oCom.moveToEnd();
                    break;
                case 36:
                    oEvt.preventDefault();
                    oCom.moveToStart();
                    break;
                case 38:
                    oEvt.preventDefault();
                    oCom.moveContentTo(-oCom.getOffset() - oCom.cfg.moveBy, false);
                    break;
                case 40:
                    oEvt.preventDefault();
                    oCom.moveContentTo(-oCom.getOffset() + oCom.cfg.moveBy, false);
                    break;
            }
        };
    }

    function onWinResize(oCom) {
        return function() {
            this.clearTimeout(nTimerResize);
            nTimerResize = this.setTimeout(function() {
                oCom.compute();
            }, 100);
        };
    }

    function resizeScrollbar(oCom) {
        oCom.faceElm[0].style.height = (oCom.faceHeight - oCom.faceBorders) + 'px';
        setOffsetTop(oCom, oCom.topContent);
    }

    var oProto = {
        boxCreate: function(oData) {
            this.rootElm = $('#moodboard');
            this.faceBorders = oData.scrollbarBorders || 0;
            this.parallax = box.get('ysl').create('parallax.moodboard', {
                rootElm: oData.parallaxRootElm,
                itemsElm: oData.parallaxItemsElm
            });
            this.parallaxMinWidth = oData.parallaxMinWidth;
            disableOnFirstTouch(this);
            this.topContent = this.topContentFrom = this.topContentTo = 0;
            this.topFace = this.topFaceFrom = this.topFaceTo = 0;
            this.timeStamp = 0;
            this.disabled = true;
            this.enable();
        },

        compute: function() {
            var $win = box.getJWin();
            this.viewWidth = $win.width();
            this.viewHeight = $win.height();
            this.pageHeight = this.rootElm.height();
            this.faceHeight = this.viewHeight / this.pageHeight * this.viewHeight;
            this.maxContentOffset = this.pageHeight - this.viewHeight;
            this.maxFaceOffset = this.viewHeight - this.faceHeight;
            if(this.faceElm) {
                resizeScrollbar(this);
            }
        },

        disable: function() {
            if(this.disabled === false) {
                this.stopParallax();
                this.disableScrolling();
                box.getJWin().unbind('resize');
                removeScrollbar(this);
                this.disabled = true;
            }
        },

        enable: function() {
            if(this.disabled === true) {
                this.compute();
                if(bTouch === false && this.viewWidth >= this.parallaxMinWidth && this.pageHeight > this.viewHeight) {
                    addScrollbar(this);
                    this.enableScrolling();
                    box.getJWin().resize(onWinResize(this));
                    this.disabled = false;
                } else {
                    $(box.getDoc().documentElement).addClass(this.cfg.clsNoSupport);
                    this.boxPublish('nosupport');
                }
            }
        },

        disableScrolling: function() {
            box.getJDoc().unbind('DOMMouseScroll mousewheel keydown.moodboard');
        },

        enableScrolling: function() {
            box.getJDoc().bind('DOMMouseScroll mousewheel', onDocScroll(this)).bind('keydown.moodboard', onDocKeyDown(this));
        },

        getOffset: function() {
            return parseInt(this.rootElm[0].style.top, 10) || 0;
        },

        getScrollFactor: function() {
            return -this.getOffset() / this.maxContentOffset;
        },

        moveScrollbarTo: function(nTop, bAnim) {
            if(nTop < 0) {
                nTop = 0;
            } else if(nTop > this.maxFaceOffset) {
                nTop = this.maxFaceOffset;
            }
            this.topFaceFrom = parseInt(this.faceElm[0].style.top, 10);
            this.topFaceTo = nTop;
            this.isMovingContent = false;
            if(bAnim === false) {
                setOffsetTop(this, nTop);
            } else {
                computeFrames(this);
            }
        },

        moveContentTo: function(nTop, bAnim) {
            if(nTop < 0) {
                nTop = 0;
            } else if(nTop > this.maxContentOffset) {
                nTop = this.maxContentOffset;
            }
            this.topContentFrom = -this.getOffset();
            this.topContentTo = nTop;
            this.isMovingContent = true;
            if(bAnim === false) {
                setOffsetTop(this, nTop);
            } else {
                computeFrames(this);
            }
        },

        moveToStart: function() {
            this.topContentFrom = -this.getOffset();
            this.topContentTo = 0;
            this.isMovingContent = true;
            computeFrames(this);
        },

        moveToEnd: function() {
            this.topContentFrom = -this.getOffset();
            this.topContentTo = this.maxContentOffset;
            this.isMovingContent = true;
            computeFrames(this);
        },

        isSupported: function() {
            return this.disabled === false && bTouch === false && this.viewWidth >= this.parallaxMinWidth;
        },

        startParallax: function() {
            if(this.isSupported()) {
                this.parallax.activate({
                    pageHeight: this.pageHeight,
                    viewHeight: this.viewHeight,
                    maxScroll: this.maxContentOffset,
                    scrollTop: this.getOffset(),
                    scrollFactor: this.getScrollFactor()
                });
            }
        },

        stopParallax: function() {
            this.parallax.deactivate();
        }
    };

    return box.get('util:component').create({
        extend: oProto
    });
});